# boxes-lib.pl
# Functions to parsing user mail files

use POSIX;

# list_mails(user, [start], [end])
sub list_mails
{
local (@rv, $h, $done);
local @index = &build_index($_[0]);
local ($start, $end);
if (@_ == 1) {
	$start = 0; $end = @index-1;
	}
elsif ($_[2] < 0) {
	$start = @index+$_[2]-1; $end = @index+$_[1]-1;
	$start = $start<0 ? 0 : $start;
	}
else {
	$start = $_[1]; $end = $_[2];
	$end = @index-1 if ($end >= @index);
	}
@rv = map { undef } @index;
open(MAIL, &user_mail_file($_[0]));
for($i=$start; $i<=$end; $i++) {
	local ($mail, $line, @headers);
	seek(MAIL, $index[$i]->[0], 0);

	# read RFC822 headers
	$mail->{'line'} = $index[$i]->[1];
	local $lnum = 0;
	while(1) {
		$lnum++;
		$line = <MAIL>;
		$mail->{'size'} += length($line);
		$line =~ s/\r|\n//g;
		last if ($line =~ /^$/);
		if ($line =~ /^(\S+):\s*(.*)/) {
			push(@headers, [ $1, $2 ]);
			}
		elsif ($line =~ /^(\s+.*)/) {
			$headers[$#headers]->[1] .= $1 unless($#headers < 0);
			}
		elsif ($line =~ /^From\s+.*\d+/) {
			$mail->{'fromline'} = $line;
			}
		}
	$mail->{'headers'} = \@headers;
	foreach $h (@headers) {
		$mail->{'header'}->{lc($h->[0])} = $h->[1];
		}

	# read the mail body
	while(1) {
		$line = <MAIL>;
		last if (!$line || $line =~ /^From\s+.*\d+\n/);
		$lnum++;
		$mail->{'size'} += length($line);
		$mail->{'body'} .= $line;
		}
	$mail->{'eline'} = $mail->{'line'} + $lnum - 1;
	$mail->{'idx'} = $i;
	$rv[$i] = $mail;
	}
return @rv;
}

# search_mail(user, field, match)
# Returns an array of messages matching some search
sub search_mail
{
local @index = &build_index($_[0]);
local (@rv, $i);
open(MAIL, &user_mail_file($_[0]));
for($i=@index-1; $i>=0; $i--) {
	local ($mail, $line, @headers);
	seek(MAIL, $index[$i]->[0], 0);

	# read mail headers
	$mail->{'line'} = $index[$i]->[1];
	local $lnum = 0;
	while(1) {
		$lnum++;
		($line = <MAIL>) =~ s/\r|\n//g;
		$mail->{'size'} += length($line);
		last if ($line =~ /^$/);
		if ($line =~ /^(\S+):\s*(.*)/) {
			push(@headers, [ $1, $2 ]);
			}
		elsif ($line =~ /^(\s+.*)/) {
			$headers[$#headers]->[1] .= $1;
			}
		elsif ($line =~ /^From\s+.*\d+/) {
			$mail->{'fromline'} = $line;
			}
		}
	$mail->{'headers'} = \@headers;
	foreach $h (@headers) {
		$mail->{'header'}->{lc($h->[0])} = $h->[1];
		}

	# read mail body
	while(1) {
		$line = <MAIL>;
		last if (!$line || $line =~ /^From\s+.*\d+\n/);
		$lnum++;
		$mail->{'size'} += length($line);
		$mail->{'body'} .= $line;
		}
	$mail->{'eline'} = $mail->{'line'} + $lnum - 1;
	$mail->{'idx'} = $i;
	if ($_[1] eq 'body') {
		push(@rv, $mail) if ($mail->{'body'} =~ /$_[2]/i);
		}
	elsif ($_[1] eq 'size') {
		push(@rv, $mail) if ($mail->{'size'} > $_[2]);
		}
	else {
		push(@rv, $mail) if ($mail->{'header'}->{$_[1]} =~ /$_[2]/i);
		}
	}
return @rv;

}

# build_index(user)
sub build_index
{
local @index;
local @ist = stat("$module_config_directory/$_[0].index");
local @st = stat(&user_mail_file($_[0]));

if (open(INDEX, "$module_config_directory/$_[0].index")) {
	@index = map { /(\d+)\s+(\d+)/; [ $1, $2 ] } <INDEX>;
	close(INDEX);
	}

if (!@ist || !@st || $ist[9] < $st[9]) {
	# The mail file is newer than the index
	local $fromok = 1;
	local ($l, $ll);
	if (open(MAIL, &user_mail_file($_[0]))) {
		local $il = $#index;
		local $i;
		for($i=($il>100 ? 100 : $il); $i>=0; $i--) {
			$l = $index[$il-$i];
			seek(MAIL, $index[$il-$i]->[0], 0);
			$ll = <MAIL>;
			$fromok = 0 if ($ll !~ /^From\s+.*\d+\n/);
			}
		}
	else { $fromok = 0; }
	local ($pos, $lnum);
	if (scalar(@index) && $fromok && $st[7] > $l->[0]) {
		# Mail file seems to have gotten bigger, most likely
		# because new mail has arrived ... only reindex the new mails
		$pos = $l->[0] + length($ll);
		$lnum = $l->[1] + 1;
		}
	else {
		# Mail file has changed in some other way ... do a rebuild
		$pos = 0;
		$lnum = 0;
		undef(@index);
		seek(MAIL, 0, 0);
		}
	while(<MAIL>) {
		if (/^From\s+.*\d+\n/) {
			push(@index, [ $pos, $lnum ]);
			}
		$pos += length($_);
		$lnum++;
		}
	close(MAIL);
	open(INDEX, ">$module_config_directory/$_[0].index");
	print INDEX map { $_->[0]." ".$_->[1]."\n" } @index;
	close(INDEX);
	}
return @index;
}

# parse_mail(&mail)
# Extracts the attachments from the mail body
sub parse_mail
{
local $ct = $_[0]->{'header'}->{'content-type'};
local (@attach, $h, $a);
if ($ct =~ /multipart\/(\S+)/i && ($ct =~ /boundary="([^"]+)"/i ||
				   $ct =~ /boundary=([^;\s]+)/i)) {
	# Multipart MIME message
	local $bound = "--".$1;
	local @lines = split(/\n/, $_[0]->{'body'});
	local $l;
	local $max = @lines;
	while($l < $max && $lines[$l++] ne $bound) {
		# skip to first boundary
		}
	while(1) {
		# read attachment headers
		local (@headers, $attach);
		while($lines[$l]) {
			if ($lines[$l] =~ /^(\S+):\s*(.*)/) {
				push(@headers, [ $1, $2 ]);
				}
			elsif ($lines[$l] =~ /^(\s+.*)/) {
				$headers[$#headers]->[1] .= $1;
				}
			$l++;
			}
		$l++;
		$attach->{'headers'} = \@headers;
		foreach $h (@headers) {
			$attach->{'header'}->{lc($h->[0])} = $h->[1];
			}
		if ($attach->{'header'}->{'content-type'} =~ /^([^;]+)/) {
			$attach->{'type'} = lc($1);
			}
		else {
			$attach->{'type'} = 'text/plain';
			}
		if ($attach->{'header'}->{'content-disposition'} =~
		    /filename="([^"]+)"/i) {
			$attach->{'filename'} = $1;
			}
		elsif ($attach->{'header'}->{'content-disposition'} =~
		       /filename=([^;\s]+)/i) {
			$attach->{'filename'} = $1;
			}
		elsif ($attach->{'header'}->{'content-type'} =~
		    /name="([^"]+)"/i) {
			$attach->{'filename'} = $1;
			}

		# read the attachment body
		while($l < $max && $lines[$l] !~ /^$bound(--)?$/) {
			$attach->{'data'} .= $lines[$l]."\n";
			$l++;
			}

		# decode if necessary
		if (lc($attach->{'header'}->{'content-transfer-encoding'}) eq
		    'base64') {
			$attach->{'data'} = &b64decode($attach->{'data'});
			}
		elsif (lc($attach->{'header'}->{'content-transfer-encoding'}) eq
		       'x-uue') {
			$attach->{'data'} = &uudecode($attach->{'data'});
			}
		elsif (lc($attach->{'header'}->{'content-transfer-encoding'}) eq
		       'quoted-printable') {
			$attach->{'data'} = &quoted_decode($attach->{'data'});
			}

		$attach->{'idx'} = scalar(@attach);
		push(@attach, $attach) if (@headers || $attach->{'data'});
		if ($attach->{'type'} eq 'message/rfc822') {
			# Decode this included email as well
			local ($amail, @aheaders, $i);
			local @alines = split(/\n/, $attach->{'data'});
			while($i < @alines && $alines[$i]) {
				if ($alines[$i] =~ /^(\S+):\s*(.*)/) {
					push(@aheaders, [ $1, $2 ]);
					}
				elsif ($alines[$i] =~ /^(\s+.*)/) {
					$aheaders[$#aheaders]->[1] .= $1;
					}
				$i++;
				}
			$amail->{'headers'} = \@aheaders;
			foreach $h (@aheaders) {
				$amail->{'header'}->{lc($h->[0])} = $h->[1];
				}
			splice(@alines, 0, $i);
			$amail->{'body'} = join("\n", @alines)."\n";
			&parse_mail($amail);
			map { $_->{'idx'} += scalar(@attach) }
			    @{$amail->{'attach'}};
			push(@attach, @{$amail->{'attach'}});
			}
		elsif ($attach->{'type'} =~ /multipart\/(\S+)/i) {
			# This attachment contains more attachments
			local $amail = { 'header' => $attach->{'header'},
					 'body' => $attach->{'data'} };
			&parse_mail($amail);
			pop(@attach);
			map { $_->{'idx'} += scalar(@attach) }
			    @{$amail->{'attach'}};
			push(@attach, @{$amail->{'attach'}});
			}
		last if ($l >= $max || $lines[$l] eq "$bound--");
		$l++;
		}
	$_[0]->{'attach'} = \@attach;
	}
elsif ($_[0]->{'body'} =~ /begin\s+([0-7]+)\s+(\S+)/i) {
	# Message contains uuencoded file(s)
	local @lines = split(/\n/, $_[0]->{'body'});
	local ($attach, $rest);
	foreach $l (@lines) {
		if ($l =~ /begin\s+([0-7]+)\s+(\S+)/i) {
			$attach = { 'type' => &guess_type($2),
				    'idx' => scalar(@{$_[0]->{'attach'}}),
				    'filename' => $2 };
			push(@{$_[0]->{'attach'}}, $attach);
			}
		elsif ($l =~ /^end/ && $attach) {
			$attach = undef;
			}
		elsif ($attach) {
			$attach->{'data'} .= unpack("u", $l);
			}
		else {
			$rest .= $l;
			}
		}
	if ($rest =~ /\S/) {
		# Some leftover text
		push(@{$_[0]->{'attach'}},
			{ 'type' => "text/plain",
			  'idx' => scalar(@{$_[0]->{'attach'}}),
			  'data' => $rest });
		}
	}
elsif ($_[0]->{'header'}->{'content-transfer-encoding'} eq 'base64') {
	# Signed body section
	$ct =~ s/;.*$//;
	$_[0]->{'attach'} = [ { 'type' => lc($ct),
				'idx' => 0,
				'data' => &b64decode($_[0]->{'body'}) } ];
	}
else {
	# One big attachment (probably text)
	local ($type, $body);
	($type = $ct) =~ s/;.*$//;
	$type = 'text/plain' if (!$type);
	if ($_[0]->{'header'}->{'content-transfer-encoding'} eq 'base64') {
		$body = &b64decode($_[0]->{'body'});
		}
	else {
		$body = $_[0]->{'body'};
		}
	$_[0]->{'attach'} = [ { 'type' => lc($type),
				'idx' => 0,
				'data' => $body } ];
	}
}

# delete_mail(user, &mail, ...)
# Delete mail messages from a user by copying the file and rebuilding the index
sub delete_mail
{
local @m = sort { $a->{'line'} <=> $b->{'line'} } @_[1..@_-1];
local $i = 0;
local $f = &user_mail_file($_[0]);
local $lnum = 0;
local %dline;
local ($dpos = 0, $dlnum = 0);
local @index;

open(SOURCE, $f);
open(DEST, ">$f.del");
while(<SOURCE>) {
	if ($i >= @m || $lnum < $m[$i]->{'line'}) {
		if (/^From\s+.*\d+\n/) {
			push(@index, [ $dpos, $dlnum ]);
			}
		$dpos += length($_);
		$dlnum++;
		local $w = (print DEST $_);
		if (!$w) {
			local $e = $?;
			close(DEST);
			close(SOURCE);
			unlink("$f.del");
			&error("Write failed : $e");
			}
		}
	elsif ($lnum == $m[$i]->{'eline'}) {
		$dline{$m[$i]->{'line'}}++;
		$i++;
		}
	$lnum++;
	}
close(SOURCE);
close(DEST);
local @st = stat($f);
unlink($f);
open(INDEX, ">$module_config_directory/$_[0].index");
print INDEX map { $_->[0]." ".$_->[1]."\n" } @index;
close(INDEX);
rename("$f.del", $f);
chown($st[4], $st[5], $f);
chmod($st[2], $f);
}

# modify_mail(user, old, new)
# Modify one email message in a mailbox by copying the file and rebuilding
# the index.
sub modify_mail
{
local $f = &user_mail_file($_[0]);
local $lnum = 0;
local ($sizediff, $linesdiff);
local @index = &build_index($_[0]);

# Replace the email that gets modified
open(SOURCE, $f);
open(DEST, ">$f.del");
while(<SOURCE>) {
	if ($lnum < $_[1]->{'line'} || $lnum > $_[1]->{'eline'}) {
		# before or after the message to change
		local $w = (print DEST $_);
		if (!$w) {
			local $e = $?;
			close(DEST);
			close(SOURCE);
			unlink("$f.del");
			&error("Write failed : $e");
			}
		}
	elsif ($lnum == $_[1]->{'line'}) {
		# found start of message to change .. put in the new one
		close(DEST);
		local @ost = stat("$f.del");
		local $nlines = &send_mail($_[2], "$f.del");
		local @nst = stat("$f.del");
		local $newsize = $nst[7] - $ost[7];
		$sizediff = $newsize - $_[1]->{'size'};
		$linesdiff = $nlines - ($_[1]->{'eline'} - $_[1]->{'line'} + 1);
		open(DEST, ">>$f.del");
		}
	$lnum++;
	}
close(SOURCE);
close(DEST);

# Now update the index and delete the temp file
foreach $i (@index) {
	if ($i->[1] > $_[1]->{'line'}) {
		# Shift mails after the modified
		$i->[0] += $sizediff;
		$i->[1] += $linesdiff;
		}
	}
local @st = stat($f);
unlink($f);
open(INDEX, ">$module_config_directory/$_[0].index");
print INDEX map { $_->[0]." ".$_->[1]."\n" } @index;
close(INDEX);
rename("$f.del", $f);
chown($st[4], $st[5], $f);
chmod($st[2], $f);

}

# send_mail(&mail, [file])
# Send out some email message or append it to a file.
# Returns the number of lines written.
sub send_mail
{
local (%header, $h);
local $lnum = 0;
foreach $h (@{$_[0]->{'headers'}}) {
	$header{lc($h->[0])} = $h->[1];
	}
local @from = &address_parts($header{'from'});
if ($_[1]) {
	# Just append the email to a file using mbox format
	local @tm = localtime(time());
	open(MAIL, ">>$_[1]");
	$lnum++;
	print MAIL $_[0]->{'fromline'} ? $_[0]->{'fromline'}."\n" :
		   strftime("From $from[0] %a %b %e %H:%M:%S %Y\n", @tm);
	push(@{$_[0]->{'headers'}},
	     [ 'Date', strftime("%a, %d %b %Y %H:%M:%S %Z", @tm) ])
		if (!$header{'date'});
	}
elsif ($config{'send_mode'}) {
	# Connect to SMTP server
	&open_socket($config{'send_mode'}, 25, MAIL);
	&smtp_command(MAIL);
	&smtp_command(MAIL, "helo ".&from_hostname()."\n");
	&smtp_command(MAIL, "mail from: $from[0]\n");
	foreach $u (&address_parts($header{'to'}.",".$header{'cc'}.
						 ",".$header{'bcc'})) {
		&smtp_command(MAIL, "rcpt to: $u\n");
		}
	&smtp_command(MAIL, "data\n");
	}
else {
	# Start sendmail
	open(MAIL, "| $config{'sendmail_path'} -t -f$from[0] >/dev/null 2>&1");
	}
foreach $h (@{$_[0]->{'headers'}}) {
	if ($h->[0] !~ /^(MIME-Version|Content-Type)$/i) {
		print MAIL $h->[0],": ",$h->[1],"\n";
		$lnum++;
		}
	}
print MAIL "MIME-Version: 1.0\n";
local $bound = "--------".time();
print MAIL "Content-Type: multipart/mixed; boundary=\"$bound\"\n";
print MAIL "\n";
$lnum += 3;

# Send attachments
print MAIL "This is a multi-part message in MIME format.\n";
$lnum++;
foreach $a (@{$_[0]->{'attach'}}) {
	print MAIL "--",$bound,"\n";
	$lnum++;
	local $enc;
	foreach $h (@{$a->{'headers'}}) {
		print MAIL $h->[0],": ",$h->[1],"\n";
		$enc = $h->[1] if (lc($h->[0]) eq 'content-transfer-encoding');
		$lnum++;
		}
	print MAIL "\n";
	$lnum++;
	if (lc($enc) eq 'base64') {
		local $enc = &encode_base64($a->{'data'});
		print MAIL $enc;
		$lnum += ($enc =~ tr/\n/\n/);
		}
	else {
		print MAIL $a->{'data'};
		$lnum += ($a->{'data'} =~ tr/\n/\n/);
		if ($a->{'data'} !~ /\n$/) {
			print MAIL "\n";
			$lnum++;
			}
		}
	}
print MAIL "--",$bound,"--\n";
$lnum++;
if ($config{'send_mode'} && !$_[1]) {
	&smtp_command(MAIL, ".\n");
	&smtp_command(MAIL, "quit\n");
	}
close(MAIL);
return $lnum;
}

# b64decode(string)
# Converts a string from base64 format to normal
sub b64decode
{
    local($str) = $_[0];
    local($res);
    $str =~ tr|A-Za-z0-9+=/||cd;
    $str =~ s/=+$//;
    $str =~ tr|A-Za-z0-9+/| -_|;
    while ($str =~ /(.{1,60})/gs) {
        my $len = chr(32 + length($1)*3/4);
        $res .= unpack("u", $len . $1 );
    }
    return $res;
}

sub guess_type
{
local $e;
if (!%mime_types) {
	open(MIME, "../mime.types");
	while(<MIME>) {
		s/\r|\n//g;
		s/#.*$//g;
		local @s = split(/\s+/);
		foreach $e (@s[1..$#s]) {
			$mime_types{$e} = $s[0];
			}
		}
	close(MIME);
	}
if ($_[0] =~ /\.([A-z0-9]+)$/ && $mime_types{$1}) {
	return $mime_types{$1};
	}
return "application/octet-stream";
}

# can_read_mail(user)
sub can_read_mail
{
return 0 if ($access{'mmode'} == 0);
return 1 if ($access{'mmode'} == 1);
return 1 if ($_[0] && $access{'sent'} eq $_[0]);
local $u;
if ($access{'mmode'} == 2) {
	foreach $u (split(/\s+/, $access{'musers'})) {
		return 1 if ($u eq $_[0]);
		}
	return 0;
	}
elsif ($access{'mmode'} == 4) {
	return 1 if ($_[0] eq $ENV{'REMOTE_USER'});
	}
elsif ($access{'mmode'} == 5) {
	local @u = getpwnam($_[0]);
	return $u[3] == $access{'musers'};
	}
elsif ($access{'mmode'} == 3) {
	foreach $u (split(/\s+/, $access{'musers'})) {
		return 0 if ($u eq $_[0]);
		}
	return 1;
	}
return 0;	# can't happen!
}

# from_hostname()
sub from_hostname
{
local ($d, $masq);
local $conf = &get_sendmailcf();
foreach $d (&find_type("D", $conf)) {
	if ($d->{'value'} =~ /^M\s*(\S*)/) { $masq = $1; }
	}
return $masq ? $masq : &get_system_hostname();
}

# mail_from_queue(qfile, dfile)
sub mail_from_queue
{
local $mail;
open(QF, $_[0]);
while(<QF>) {
	s/\r|\n//g;
	if (/^H\?[^\?]*\?(\S+):\s+(.*)/ || /^H(\S+):\s+(.*)/) {
		push(@headers, [ $1, $2 ]);
		}
	elsif (/^(\s+.*)/) {
		$headers[$#headers]->[1] .= $1;
		}
	}
close(QF);
$mail->{'headers'} = \@headers;
foreach $h (@headers) {
	$mail->{'header'}->{lc($h->[0])} = $h->[1];
	}

# Read the mail body
open(DF, $_[1]);
while(<DF>) {
	$mail->{'body'} .= $_;
	}
close(DF);
return $mail;
}

# wrap_lines(text, width)
# Given a multi-line string, return an array of lines wrapped to
# the given width
sub wrap_lines
{
local @rv;
local $w = $_[1];
foreach $rest (split(/\n/, $_[0])) {
	if ($rest =~ /\S/) {
		while($rest =~ /^(.{1,$w}\S*)\s*([\0-\377]*)$/) {
			push(@rv, $1);
			$rest = $2;
			}
		}
	else {
		# Empty line .. keep as it is
		push(@rv, $rest);
		}
	}
return @rv;
}

# smtp_command(handle, command)
sub smtp_command
{
local ($m, $c) = @_;
print $m $c;
local $r = <$m>;
if ($r !~ /^[23]\d+/) {
	&error(&text('send_esmtp', "<tt>$c</tt>", "<tt>$r</tt>"));
	}
}

# address_parts(string)
sub address_parts
{
local @rv;
local $rest = $_[0];
while($rest =~ /([^<>\s,'"\@]+\@[A-z0-9\-\.\!]+)(.*)/) {
	push(@rv, $1);
	$rest = $2;
	}
return @rv;
}

# link_urls(text)
sub link_urls
{
local $r = $_[0];
$r =~ s/((http|ftp|https|mailto):[^><"'\s]+)/<a href="$1">$1<\/a>/g;
return $r;
}

# uudecode(text)
sub uudecode
{
local @lines = split(/\n/, $_[0]);
local ($l, $data);
for($l=0; $lines[$l] !~ /begin\s+([0-7]+)\s/i; $l++) { }
while($lines[++$l]) {
	$data .= unpack("u", $lines[$l]);
	}
return $data;
}

sub simplify_date
{
if ($_[0] =~ /^(\S+),\s+0*(\d+)\s+(\S+)\s+(\d+)\s+(\d+):(\d+)/) {
	return "$2/$3/$4 $5:$6";
	}
return $_[0];
}

# simplify_from(from)
# Simplifies a From: address for display in the list mail
sub simplify_from
{
local $rv = &decode_mimewords($_[0]);
if ($rv !~ /\S/) {
	return $text{'mail_nonefrom'};
	}
elsif ($rv =~ /(.*\S.*)<[^>]+>/) {
	return &html_escape("$1");
	}
else {
	return &html_escape($rv);
	}
}

# simplify_subject(subject)
sub simplify_subject
{
local $rv = &decode_mimewords($_[0]);
return $rv =~ /\S/ ? &html_escape($rv) : "<br>";
}

# quoted_decode(text)
sub quoted_decode
{
local $t = $_[0];
$t =~ s/=\n//g;
$t =~ s/=(\S\S)/pack("c",hex($1))/ge;
return $t;
}

sub decode_mimewords {
    my $encstr = shift;
    my %params = @_;
    my @tokens;
    $@ = '';           ### error-return

    ### Collapse boundaries between adjacent encoded words:
    $encstr =~ s{(\?\=)\r?\n[ \t](\=\?)}{$1$2}gs;
    pos($encstr) = 0;
    ### print STDOUT "ENC = [", $encstr, "]\n";

    ### Decode:
    my ($charset, $encoding, $enc, $dec);
    while (1) {
	last if (pos($encstr) >= length($encstr));
	my $pos = pos($encstr);               ### save it

	### Case 1: are we looking at "=?..?..?="?
	if ($encstr =~    m{\G             # from where we left off..
			    =\?([^?]*)     # "=?" + charset +
			     \?([bq])      #  "?" + encoding +
			     \?([^?]+)     #  "?" + data maybe with spcs +
			     \?=           #  "?="
			    }xgi) {
	    ($charset, $encoding, $enc) = ($1, lc($2), $3);
	    $dec = (($encoding eq 'q') ? _decode_Q($enc) : _decode_B($enc));
	    push @tokens, [$dec, $charset];
	    next;
	}

	### Case 2: are we looking at a bad "=?..." prefix? 
	### We need this to detect problems for case 3, which stops at "=?":
	pos($encstr) = $pos;               # reset the pointer.
	if ($encstr =~ m{\G=\?}xg) {
	    $@ .= qq|unterminated "=?..?..?=" in "$encstr" (pos $pos)\n|;
	    push @tokens, ['=?'];
	    next;
	}

	### Case 3: are we looking at ordinary text?
	pos($encstr) = $pos;               # reset the pointer.
	if ($encstr =~ m{\G                # from where we left off...
			 ([\x00-\xFF]*?    #   shortest possible string,
			  \n*)             #   followed by 0 or more NLs,
		         (?=(\Z|=\?))      # terminated by "=?" or EOS
			}xg) {
	    length($1) or die "MIME::Words: internal logic err: empty token\n";
	    push @tokens, [$1];
	    next;
	}

	### Case 4: bug!
	die "MIME::Words: unexpected case:\n($encstr) pos $pos\n\t".
	    "Please alert developer.\n";
    }
    return join('',map {$_->[0]} @tokens);
}

# _decode_Q STRING
#     Private: used by _decode_header() to decode "Q" encoding, which is
#     almost, but not exactly, quoted-printable.  :-P
sub _decode_Q {
    my $str = shift;
    $str =~ s/_/\x20/g;                                # RFC-1522, Q rule 2
    $str =~ s/=([\da-fA-F]{2})/pack("C", hex($1))/ge;  # RFC-1522, Q rule 1
    $str;
}

# _decode_B STRING
#     Private: used by _decode_header() to decode "B" encoding.
sub _decode_B {
    my $str = shift;
    &decode_base64($str);
}

# user_mail_file(user)
sub user_mail_file
{
if ($config{'mail_dir'}) {
	return "$config{'mail_dir'}/$_[0]";
	}
else {
	local @u = getpwnam($_[0]);
	return "$u[7]/mbox";
	}
}

1;
