# solaris-lib.pl
# Functions for solaris/hpux-style printer management

# list_printers()
# Returns an array of known printer names
sub list_printers
{
return () if (!&sched_running());
local(@rv, $_);
open(STAT, "lpstat -v |");
while(<STAT>) {
	if (/\s+for\s+(\S+):/i) { push(@rv, $1); }
	}
close(STAT);
return &unique(@rv);
}

# get_printer(name, [nostatus])
# Returns a reference to an associative array of printer details
sub get_printer
{
local($stat, @rv, $body, $avl, $con, $sys, %prn, $_, $out);
$out = `lpstat -l -p $_[0]`;
if ($out =~ /^printer\s+(\S+)\s+(\S+).*\n([\000-\177]*)$/) {
	# printer exists..
	$prn{'name'} = $1;
	$prn{'enabled'} = $2 ne "disabled";
	$body = $3;
	}
else {
	# no printer found
	return undef;
	}
if (!$prn{'enabled'} && $body =~ /^\s+(.*)/) {
	$prn{'enabled_why'} = lc($1) eq "paused" || lc($1) eq "reason unknown" ?
			      "" : $1;
	}
if ($body =~ /Description: (.*)/) { $prn{'desc'} = $1; }
if ($body =~ /Printer types: (.*)/) { $prn{'ptype'} = $1; }
if ($body =~ /Interface: (.*)/ && $1 ne $config{'default_interface'})
	{ $prn{'iface'} = $1; }
if ($body =~ /Banner not required/) { $prn{'banner'} = 0; }
else { $prn{'banner'} = 1; }

if (!$_[1]) {
	# request availability
	$avl = `lpstat -a $prn{'name'} 2>&1`;
	if ($avl =~ /^\S+\s+not accepting.*\n\s*(.*)/) {
		$prn{'accepting'} = 0;
		$prn{'accepting_why'} = lc($1) eq "reason unknown" ? "" : $1;
		}
	else { $prn{'accepting'} = 1; }
	}

# request connection
$con = `lpstat -v $prn{'name'} 2>&1`;
if ($con =~ /^device for \S+:\s+(lpd|ipp):\/\/(\S+)\/(\S+)/) {
	$prn{'rhost'} = $2;
	$prn{'rqueue'} = $3;
	if ($1 eq 'ipp') {
		$prn{'rtype'} = 'ipp';
		$prn{'rhost'} =~ s/:631//;
		}
	}
elsif ($con =~ /^device for \S+:\s+socket:\/\/(\S+):(\d+)/) {
	$prn{'dhost'} = $1;
	$prn{'dport'} = $2;
	}
elsif ($con =~ /^device for \S+:\s+(file|serial|parallel):([^\s\?]+)/) {
	$prn{'dev'} = $2;
	}
elsif ($con =~ /^device for \S+:\s+(\S+)/) {
	$prn{'dev'} = $1;
	}

# Check if this is the default printer
`lpstat -d 2>&1` =~ /destination: (\S+)/;
if ($1 eq $prn{'name'}) { $prn{'default'} = 1; }

return \%prn;
}

# get_jobs(printer)
sub get_jobs
{
local (@jobs, $htype);
open(LPQ, "lpq -P$_[0] |");
while(<LPQ>) {
	chop;
	if (/^(\S+)\s+(\S+)\s+(\d+)\s+(.*\S)\s+(\d+)\s+(\S+)$/) {
		# Normal lpq output
		local(%job, $f, @pq);
		$job{'id'} = $3;
		$job{'user'} = $2;
		$job{'size'} = $5;
		$job{'file'} = $4;
		$job{'printing'} = ($1 eq "active");
		local $d = $config{'spool_dir'};
		opendir(DIR, $d);
		while($f = readdir(DIR)) {
			if ($f =~ /^d(\d+)\-(\d+)$/ && $1 == $job{'id'}) {
				push(@pq, "$d/$f");
				}
			}
		closedir(DIR);
		$job{'printfile'} = @pq ? \@pq : undef;
		push(@jobs, \%job);
		}
	}
close(LPQ);
return @jobs;
}

# printer_support(option)
sub printer_support
{
return $_[0] !~ /^(msize|alias|rnoqueue|ctype|sysv|allow)$/;
}

# list_classes()
# Returns an associative array of print classes
sub list_classes
{
local($stat, %rv);
$stat = `lpstat -c 2>&1`;
while($stat =~ /^members of class (\S+):\n((\s+\S+\n)+)([\000-\177]*)$/) {
	$stat = $4;
	$rv{$1} = [ grep { $_ ne "" } split(/\s+/, $2) ];
	}
return \%rv;
}

# create_printer(&details)
# Create a new printer
sub create_printer
{
&modify_printer($_[0]);
}

# modify_printer(&details)
# Change an existing printer
sub modify_printer
{
local(%prn, $cmd, $out);
%prn = %{$_[0]};

# call lpadmin
$cmd = "lpadmin -p $prn{'name'} -D \"$prn{'desc'}\"";
if ($prn{'dev'} =~ /\/dev\/tty/) {
	$cmd .= " -v serial:$prn{'dev'}";
	}
elsif ($prn{'dev'} =~ /\/dev\/lp/) {
	$cmd .= " -v parallel:$prn{'dev'}";
	}
elsif ($prn{'dev'}) {
	$cmd .= " -v $prn{'dev'}";
	}
elsif ($prn{'rhost'}) {
	if ($prn{'rtype'} eq 'ipp') {
		$cmd .= " -v ipp://$prn{'rhost'}/$prn{'rqueue'}";
		}
	else {
		$cmd .= " -v lpd://$prn{'rhost'}/$prn{'rqueue'}";
		}
	}
else {
	$cmd .= " -v socket://$prn{'dhost'}:$prn{'dport'}";
	}
if ($prn{'iface'}) {
	$cmd .= " -m $prn{'iface'}";
	}
$out = &backquote_logged("cd / ; $cmd 2>&1");
if ($?) { &error("lpadmin failed : <pre>$out ($cmd)</pre>"); }

# make the default
if ($prn{'default'}) {
	$out = &backquote_logged("cd / ; lpadmin -d $prn{'name'} 2>&1");
	if ($?) { &error("Failed to set default : <pre>$out</pre>"); }
	}

# enable or disable
local $enable = &has_command("enable");
if ($prn{'enabled'}) { $cmd = "$enable $prn{'name'}"; }
elsif ($prn{'enabled_why'})
	{ $cmd = "$enable $prn{'name'} ; disable -r \"$prn{'enabled_why'}\" $prn{'name'}"; }
else { $cmd = "$enable $prn{'name'} ; disable $prn{'name'}"; }
$out = &backquote_logged("$cmd 2>&1");

# accepting or rejecting requests
if ($prn{'accepting'}) { $cmd = "accept $prn{'name'}"; }
elsif ($prn{'accepting_why'}) {
	$cmd = "accept $prn{'name'} ; ".
	       "reject -r \"$prn{'accepting_why'}\" $prn{'name'}";
	}
else { $cmd = "accept $prn{'name'} ; reject $prn{'name'}"; }
$out = &backquote_logged("$cmd 2>&1");
}

# delete_printer(name)
# Deletes some existing printer
sub delete_printer
{
local($out);
$out = &backquote_logged("lpadmin -x $_[0] 2>&1");
if ($?) { &error("lpadmin failed : <pre>$out</pre>"); }
}

# cancel_job(printer, id)
# Cancels some print job
sub cancel_job
{
local($out);
$out = &backquote_logged("lprm -P$_[0] $_[1] 2>&1");
if ($?) { &error("lprm failed : <pre>$out</pre>"); }
sleep(1);
}

# sched_running()
# Returns the pid if lpsched is running, 0 if not, -1 if cannot be stopped
sub sched_running
{
@pid = &find_byname("cups");
if (@pid) { return $pid[0]; }
return 0;
}

# start_sched()
# Start lpsched
sub start_sched
{
local $out = &backquote_logged("cupsd 2>&1");
if ($?) { &error("failed to start cups : <tt>$out</tt>"); }
sleep(3);
}

# stop_sched(pid)
# Stop the running lpsched process
sub stop_sched
{
&kill_logged('TERM', $_[0]) || &error("Failed to stop cups : $!");
}

# print_command(printer, file)
# Returns the command to print some file on some printer
sub print_command
{
return "lpr -P$_[0] $_[1]";
}

if (-r "/dev/lp0") {
	@device_files = ("/dev/lp0", "/dev/lp1",
			 "/dev/ttyS0", "/dev/ttyS1", "/dev/null");
	}
else {
	@device_files = ("/dev/lp1", "/dev/lp2",
			 "/dev/ttyS0", "/dev/ttyS1", "/dev/null");
	}
if (-r "/dev/usblp0") {
	push(@device_files, "/dev/usblp0", "/dev/usblp1",
			    "/dev/usblp2", "/dev/usblp3");
	}
@device_names = (&text('linux_paralel', "1"), &text('linux_paralel', "2"),
		 &text('linux_serial', "1"), &text('linux_serial', "2"),
		 $text{'linux_null'},  &text('linux_usb', 1),
		 &text('linux_usb', 2), &text('linux_usb', 3),
		 &text('linux_usb', 4));

